<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Vehicle extends Model
{
    protected $fillable = [
        'driver_id',
        'vehicle_type',
        'brand',
        'model',
        'year',
        'color',
        'license_plate',
        'registration_number',
        'total_seats',
        'vehicle_images',
        'is_active',
        'is_verified',
    ];

    protected function casts(): array
    {
        return [
            'year' => 'integer',
            'total_seats' => 'integer',
            'vehicle_images' => 'array',
            'is_active' => 'boolean',
            'is_verified' => 'boolean',
        ];
    }

    // Relationships

    /**
     * Get the driver who owns this vehicle
     */
    public function driver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'driver_id');
    }

    /**
     * Get all rides using this vehicle
     */
    public function rides(): HasMany
    {
        return $this->hasMany(Ride::class);
    }

    // Helper Methods

    /**
     * Get the full vehicle name (brand + model + year)
     */
    public function getFullNameAttribute(): string
    {
        return "{$this->brand} {$this->model} ({$this->year})";
    }

    /**
     * Get verification status badge
     */
    public function getVerificationBadgeAttribute(): string
    {
        return $this->is_verified ? '✓ Verified' : '⚠ Pending';
    }
}
