@extends('layouts.app')

@section('title', 'Confirm Booking')

@section('styles')
    <style>
        .booking-card {
            background: white;
            border-radius: 20px;
            padding: 20px;
            margin-bottom: 16px;
            box-shadow: 0 2px 15px rgba(0,0,0,0.05);
        }

        .card-title {
            font-size: 14px;
            font-weight: 700;
            color: #1e3a5f;
            margin-bottom: 16px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .card-title svg {
            width: 18px;
            height: 18px;
            color: #00aff5;
        }

        /* Ride Info */
        .ride-header {
            display: flex;
            gap: 14px;
            margin-bottom: 16px;
        }

        .ride-image {
            width: 80px;
            height: 60px;
            border-radius: 12px;
            background: linear-gradient(135deg, #1e3a5f, #0d1b2a);
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
        }

        .ride-image img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .ride-image svg {
            width: 30px;
            height: 30px;
            color: #00aff5;
        }

        .ride-info {
            flex: 1;
        }

        .ride-route {
            font-size: 16px;
            font-weight: 800;
            color: #1e3a5f;
            margin-bottom: 4px;
        }

        .ride-datetime {
            font-size: 12px;
            color: #64748b;
        }

        /* Route Visual */
        .route-visual {
            background: #f8fafc;
            border-radius: 14px;
            padding: 16px;
            margin-bottom: 16px;
        }

        .route-point {
            display: flex;
            align-items: flex-start;
            gap: 12px;
            position: relative;
        }

        .route-point:first-child {
            margin-bottom: 20px;
        }

        .route-point:first-child::after {
            content: '';
            position: absolute;
            left: 9px;
            top: 24px;
            width: 2px;
            height: calc(100% + 12px);
            background: linear-gradient(to bottom, #00aff5, #10b981);
        }

        .route-dot {
            width: 20px;
            height: 20px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }

        .route-dot.from {
            background: #00aff5;
        }

        .route-dot.to {
            background: #10b981;
        }

        .route-dot svg {
            width: 10px;
            height: 10px;
            color: white;
        }

        .route-details {
            flex: 1;
        }

        .route-city {
            font-size: 14px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .route-address {
            font-size: 11px;
            color: #64748b;
        }

        .route-time {
            font-size: 12px;
            font-weight: 700;
            color: #00aff5;
        }

        /* Driver Info */
        .driver-row {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 14px;
            background: #f8fafc;
            border-radius: 12px;
        }

        .driver-avatar {
            width: 50px;
            height: 50px;
            border-radius: 14px;
            background: linear-gradient(135deg, #00aff5, #10b981);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            font-weight: 800;
            color: white;
        }

        .driver-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 14px;
        }

        .driver-info {
            flex: 1;
        }

        .driver-name {
            font-size: 14px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .driver-rating {
            font-size: 12px;
            color: #f59e0b;
        }

        .driver-badge {
            background: #d1fae5;
            color: #065f46;
            font-size: 10px;
            font-weight: 700;
            padding: 4px 10px;
            border-radius: 20px;
        }

        /* Vehicle Info */
        .vehicle-row {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            background: #f8fafc;
            border-radius: 12px;
            margin-top: 10px;
        }

        .vehicle-info {
            flex: 1;
        }

        .vehicle-name {
            font-size: 13px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .vehicle-details {
            font-size: 11px;
            color: #64748b;
        }

        /* Amenities */
        .amenities-list {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-top: 12px;
        }

        .amenity-tag {
            background: #e0f7fa;
            color: #0369a1;
            font-size: 10px;
            font-weight: 600;
            padding: 6px 10px;
            border-radius: 8px;
        }

        /* Seats Selection */
        .seats-selector {
            display: flex;
            gap: 10px;
            margin-bottom: 12px;
        }

        .seat-option {
            flex: 1;
            padding: 14px;
            text-align: center;
            background: #f8fafc;
            border-radius: 12px;
            cursor: pointer;
            border: 2px solid transparent;
            transition: all 0.2s;
        }

        .seat-option:has(input:checked) {
            background: #e0f7fa;
            border-color: #00aff5;
        }

        .seat-option input {
            display: none;
        }

        .seat-number {
            font-size: 18px;
            font-weight: 800;
            color: #1e3a5f;
        }

        .seat-label {
            font-size: 10px;
            color: #64748b;
        }

        /* Price Breakdown */
        .price-row {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid #f1f5f9;
            font-size: 13px;
        }

        .price-row:last-child {
            border-bottom: none;
        }

        .price-label {
            color: #64748b;
        }

        .price-value {
            font-weight: 700;
            color: #1e3a5f;
        }

        .price-total {
            font-size: 16px;
            font-weight: 800;
            padding-top: 12px;
            border-top: 2px solid #1e3a5f;
            margin-top: 8px;
        }

        .price-total .price-value {
            color: #10b981;
            font-size: 20px;
        }

        /* Form */
        .form-group {
            margin-bottom: 14px;
        }

        .form-label {
            display: block;
            font-size: 11px;
            font-weight: 700;
            color: #64748b;
            text-transform: uppercase;
            margin-bottom: 6px;
        }

        .form-input, .form-textarea {
            width: 100%;
            padding: 14px;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            font-size: 14px;
            font-weight: 600;
            color: #1e3a5f;
            background: #f8fafc;
        }

        .form-textarea {
            min-height: 80px;
            resize: none;
        }

        .form-input:focus, .form-textarea:focus {
            outline: none;
            border-color: #00aff5;
            background: white;
        }

        /* Payment Options */
        .payment-option {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 14px;
            background: #f8fafc;
            border-radius: 12px;
            margin-bottom: 8px;
            cursor: pointer;
            border: 2px solid transparent;
        }

        .payment-option:has(input:checked) {
            background: #e0f7fa;
            border-color: #00aff5;
        }

        .payment-option input {
            display: none;
        }

        .payment-icon {
            font-size: 20px;
        }

        .payment-info {
            flex: 1;
        }

        .payment-name {
            font-size: 13px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .payment-desc {
            font-size: 10px;
            color: #64748b;
        }

        /* Submit Button */
        .submit-btn {
            display: block;
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, #00aff5, #10b981);
            color: white;
            border: none;
            border-radius: 14px;
            font-size: 15px;
            font-weight: 700;
            cursor: pointer;
            margin-top: 16px;
        }

        .submit-btn:active {
            transform: scale(0.98);
        }

        .cancel-link {
            display: block;
            text-align: center;
            margin-top: 12px;
            color: #64748b;
            font-size: 13px;
            text-decoration: none;
        }

        .alert-success {
            background: #d1fae5;
            color: #065f46;
            padding: 12px;
            border-radius: 10px;
            font-size: 12px;
            font-weight: 600;
            margin-bottom: 12px;
        }
    </style>
@endsection

@section('content')
    @if(session('success'))
        <div class="alert-success">✓ {{ session('success') }}</div>
    @endif

    <form action="{{ route('bookings.store') }}" method="POST">
        @csrf
        <input type="hidden" name="ride_id" value="{{ $ride->id }}">

        <!-- Ride Summary -->
        <div class="booking-card">
            <h2 class="card-title">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <circle cx="12" cy="10" r="3"></circle>
                    <path d="M12 21.7C17.3 17 20 13 20 10a8 8 0 1 0-16 0c0 3 2.7 6.9 8 11.7z"></path>
                </svg>
                Your Ride
            </h2>

            <div class="ride-header">
                <div class="ride-image">
                    @if($ride->vehicle->vehicle_images && count($ride->vehicle->vehicle_images) > 0)
                        <img src="{{ asset('storage/' . $ride->vehicle->vehicle_images[0]) }}" alt="{{ $ride->vehicle->brand }}">
                    @else
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                            <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9C18.7 10.6 16 10 16 10s-1.3-1.4-2.2-2.3c-.5-.4-1.1-.7-1.8-.7H5c-.6 0-1.1.4-1.4.9l-1.4 2.9A3.7 3.7 0 0 0 2 12v4c0 .6.4 1 1 1h2"></path>
                            <circle cx="7" cy="17" r="2"></circle>
                            <circle cx="17" cy="17" r="2"></circle>
                        </svg>
                    @endif
                </div>
                <div class="ride-info">
                    <div class="ride-route">{{ $ride->route->fromCity->name }} → {{ $ride->route->toCity->name }}</div>
                    <div class="ride-datetime">📅 {{ $ride->departure_datetime->format('D, d M Y') }} • ⏰ {{ $ride->departure_datetime->format('h:i A') }}</div>
                </div>
            </div>

            <div class="route-visual">
                <div class="route-point">
                    <div class="route-dot from">
                        <svg viewBox="0 0 24 24" fill="currentColor"><circle cx="12" cy="12" r="4"></circle></svg>
                    </div>
                    <div class="route-details">
                        <div class="route-city">{{ $ride->route->fromCity->name }}</div>
                        <div class="route-address">{{ $ride->pickup_point ?? 'City Center' }}</div>
                    </div>
                    <div class="route-time">{{ $ride->departure_datetime->format('h:i A') }}</div>
                </div>
                <div class="route-point">
                    <div class="route-dot to">
                        <svg viewBox="0 0 24 24" fill="currentColor"><circle cx="12" cy="12" r="4"></circle></svg>
                    </div>
                    <div class="route-details">
                        <div class="route-city">{{ $ride->route->toCity->name }}</div>
                        <div class="route-address">{{ $ride->drop_point ?? 'City Center' }}</div>
                    </div>
                    <div class="route-time">{{ $ride->arrival_datetime->format('h:i A') }}</div>
                </div>
            </div>
        </div>

        <!-- Driver & Vehicle -->
        <div class="booking-card">
            <h2 class="card-title">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                    <circle cx="12" cy="7" r="4"></circle>
                </svg>
                Driver & Vehicle
            </h2>

            <div class="driver-row">
                <div class="driver-avatar">
                    @if($ride->driver->avatar)
                        <img src="{{ asset('storage/' . $ride->driver->avatar) }}" alt="{{ $ride->driver->name }}">
                    @else
                        {{ substr($ride->driver->name, 0, 1) }}
                    @endif
                </div>
                <div class="driver-info">
                    <div class="driver-name">{{ $ride->driver->name }}</div>
                    <div class="driver-rating">⭐ {{ number_format($ride->driver->rating ?? 4.5, 1) }} • {{ $ride->driver->total_rides ?? 0 }} rides</div>
                </div>
                @if($ride->driver->is_verified)
                    <span class="driver-badge">✓ Verified</span>
                @endif
            </div>

            <div class="vehicle-row">
                <div class="vehicle-info">
                    <div class="vehicle-name">{{ $ride->vehicle->brand }} {{ $ride->vehicle->model }}</div>
                    <div class="vehicle-details">{{ $ride->vehicle->year }} • {{ $ride->vehicle->color }} • {{ $ride->vehicle->license_plate }}</div>
                </div>
            </div>

            @if($ride->amenities && count($ride->amenities) > 0)
                <div class="amenities-list">
                    @foreach($ride->amenities as $amenity)
                        <span class="amenity-tag">{{ $amenity }}</span>
                    @endforeach
                </div>
            @endif
        </div>

        <!-- Seats -->
        <div class="booking-card">
            <h2 class="card-title">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <rect x="4" y="4" width="16" height="16" rx="2"></rect>
                    <path d="M9 9h6v6H9z"></path>
                </svg>
                Select Seats
            </h2>

            <div class="seats-selector">
                @for($i = 1; $i <= min($ride->available_seats, 4); $i++)
                    <label class="seat-option">
                        <input type="radio" name="seats" value="{{ $i }}" {{ $i == 1 ? 'checked' : '' }}>
                        <div class="seat-number">{{ $i }}</div>
                        <div class="seat-label">{{ $i == 1 ? 'Seat' : 'Seats' }}</div>
                    </label>
                @endfor
            </div>
        </div>

        <!-- Contact Info -->
        <div class="booking-card">
            <h2 class="card-title">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72c.127.96.361 1.903.7 2.81a2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45c.907.339 1.85.573 2.81.7A2 2 0 0 1 22 16.92z"></path>
                </svg>
                Your Details
            </h2>

            <div class="form-group">
                <label class="form-label">Phone Number</label>
                <input type="tel" name="phone" class="form-input" placeholder="+92 300 1234567" required>
            </div>

            <div class="form-group">
                <label class="form-label">Pickup Note (Optional)</label>
                <textarea name="notes" class="form-textarea" placeholder="Any special requests or pickup instructions..."></textarea>
            </div>
        </div>

        <!-- Payment -->
        <div class="booking-card">
            <h2 class="card-title">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <rect x="1" y="4" width="22" height="16" rx="2"></rect>
                    <line x1="1" y1="10" x2="23" y2="10"></line>
                </svg>
                Payment Method
            </h2>

            <label class="payment-option">
                <input type="radio" name="payment" value="cash" checked>
                <span class="payment-icon">💵</span>
                <div class="payment-info">
                    <div class="payment-name">Cash</div>
                    <div class="payment-desc">Pay driver in cash</div>
                </div>
            </label>

            <label class="payment-option">
                <input type="radio" name="payment" value="jazzcash">
                <span class="payment-icon">📱</span>
                <div class="payment-info">
                    <div class="payment-name">JazzCash</div>
                    <div class="payment-desc">Mobile wallet</div>
                </div>
            </label>

            <label class="payment-option">
                <input type="radio" name="payment" value="easypaisa">
                <span class="payment-icon">💳</span>
                <div class="payment-info">
                    <div class="payment-name">Easypaisa</div>
                    <div class="payment-desc">Mobile wallet</div>
                </div>
            </label>
        </div>

        <!-- Price Summary -->
        <div class="booking-card">
            <h2 class="card-title">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <line x1="12" y1="1" x2="12" y2="23"></line>
                    <path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"></path>
                </svg>
                Price Summary
            </h2>

            <div class="price-row">
                <span class="price-label">Price per seat</span>
                <span class="price-value">Rs {{ number_format($ride->price_per_seat * 280, 0) }}</span>
            </div>
            <div class="price-row">
                <span class="price-label">Number of seats</span>
                <span class="price-value" id="seatCount">1</span>
            </div>
            <div class="price-row">
                <span class="price-label">Service fee</span>
                <span class="price-value">Rs 50</span>
            </div>
            <div class="price-row price-total">
                <span class="price-label">Total</span>
                <span class="price-value" id="totalPrice">Rs {{ number_format($ride->price_per_seat * 280 + 50, 0) }}</span>
            </div>
        </div>

        <button type="submit" class="submit-btn">✓ Confirm Booking</button>
        <a href="{{ url()->previous() }}" class="cancel-link">Cancel</a>
    </form>

    <script>
        const pricePerSeat = {{ $ride->price_per_seat * 280 }};
        const serviceFee = 50;
        
        document.querySelectorAll('input[name="seats"]').forEach(input => {
            input.addEventListener('change', function() {
                const seats = parseInt(this.value);
                document.getElementById('seatCount').textContent = seats;
                const total = (pricePerSeat * seats) + serviceFee;
                document.getElementById('totalPrice').textContent = 'Rs ' + total.toLocaleString();
            });
        });
    </script>
@endsection
