@extends('layouts.app')

@section('title', 'Find a Ride')

@section('styles')
    <style>
        /* Search Header */
        .search-header {
            background: linear-gradient(135deg, #1e3a5f 0%, #0d1b2a 100%);
            border-radius: 20px;
            padding: 20px;
            margin-bottom: 16px;
            color: white;
        }

        .search-title {
            font-size: 20px;
            font-weight: 800;
            margin-bottom: 4px;
        }

        .search-subtitle {
            font-size: 12px;
            color: rgba(255,255,255,0.7);
        }

        /* Search Form */
        .search-card {
            background: white;
            border-radius: 16px;
            padding: 16px;
            margin-bottom: 16px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.04);
        }

        .search-form {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }

        .search-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
        }

        .search-row.triple {
            grid-template-columns: 1fr 1fr 1fr;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }

        .form-label {
            font-size: 10px;
            font-weight: 700;
            color: #64748b;
            text-transform: uppercase;
        }

        .form-input, .form-select {
            padding: 12px;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 13px;
            font-weight: 600;
            color: #1e3a5f;
            background: #f8fafc;
        }

        .form-input:focus, .form-select:focus {
            outline: none;
            border-color: #00aff5;
            background: white;
        }

        .search-btn {
            padding: 14px;
            background: linear-gradient(135deg, #00aff5, #10b981);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 14px;
            font-weight: 700;
            cursor: pointer;
        }

        .search-btn:active {
            transform: scale(0.98);
        }

        /* Filters */
        .filter-row {
            display: flex;
            gap: 8px;
            margin-bottom: 16px;
            overflow-x: auto;
            padding-bottom: 4px;
        }

        .filter-chip {
            display: flex;
            align-items: center;
            gap: 6px;
            padding: 8px 14px;
            background: white;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            color: #64748b;
            white-space: nowrap;
            border: 2px solid #e2e8f0;
            cursor: pointer;
        }

        .filter-chip.active {
            background: #e0f7fa;
            border-color: #00aff5;
            color: #00aff5;
        }

        /* Results Header */
        .results-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
        }

        .results-count {
            font-size: 14px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .sort-btn {
            display: flex;
            align-items: center;
            gap: 4px;
            padding: 8px 12px;
            background: white;
            border-radius: 10px;
            font-size: 11px;
            font-weight: 600;
            color: #64748b;
            border: none;
            cursor: pointer;
        }

        /* Ride Card */
        .ride-card {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            margin-bottom: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.04);
            text-decoration: none;
            display: block;
        }

        .ride-card:active {
            transform: scale(0.99);
        }

        .ride-image {
            height: 120px;
            background: linear-gradient(135deg, #1e3a5f, #0d1b2a);
            position: relative;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .ride-image img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .ride-image svg {
            width: 50px;
            height: 50px;
            color: #00aff5;
        }

        .ride-price-badge {
            position: absolute;
            top: 10px;
            right: 10px;
            background: white;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 13px;
            font-weight: 800;
            color: #10b981;
        }

        .ride-date-badge {
            position: absolute;
            top: 10px;
            left: 10px;
            background: rgba(0,0,0,0.6);
            backdrop-filter: blur(10px);
            padding: 6px 10px;
            border-radius: 8px;
            font-size: 10px;
            font-weight: 700;
            color: white;
        }

        .ride-content {
            padding: 14px;
        }

        /* Route */
        .ride-route {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 12px;
        }

        .route-point {
            flex: 1;
        }

        .route-city {
            font-size: 14px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .route-time {
            font-size: 11px;
            color: #64748b;
        }

        .route-arrow {
            display: flex;
            flex-direction: column;
            align-items: center;
            color: #00aff5;
        }

        .route-arrow svg {
            width: 18px;
            height: 18px;
        }

        .route-duration {
            font-size: 9px;
            color: #64748b;
            font-weight: 600;
        }

        /* Driver Row */
        .driver-row {
            display: flex;
            align-items: center;
            gap: 10px;
            padding-top: 12px;
            border-top: 1px solid #f1f5f9;
        }

        .driver-avatar {
            width: 40px;
            height: 40px;
            border-radius: 12px;
            background: linear-gradient(135deg, #00aff5, #10b981);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 16px;
            font-weight: 800;
            color: white;
            overflow: hidden;
        }

        .driver-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .driver-info {
            flex: 1;
        }

        .driver-name {
            font-size: 13px;
            font-weight: 700;
            color: #1e3a5f;
        }

        .driver-meta {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 11px;
        }

        .driver-rating {
            color: #f59e0b;
        }

        .driver-verified {
            color: #10b981;
        }

        .seats-badge {
            background: #e0f7fa;
            color: #0369a1;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 700;
        }

        /* Amenities */
        .amenities-row {
            display: flex;
            gap: 6px;
            margin-top: 10px;
            flex-wrap: wrap;
        }

        .amenity-tag {
            font-size: 10px;
            padding: 4px 8px;
            background: #f1f5f9;
            border-radius: 6px;
            color: #64748b;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            background: white;
            border-radius: 16px;
        }

        .empty-icon {
            font-size: 50px;
            margin-bottom: 16px;
        }

        .empty-title {
            font-size: 18px;
            font-weight: 800;
            color: #1e3a5f;
            margin-bottom: 8px;
        }

        .empty-text {
            font-size: 13px;
            color: #64748b;
            margin-bottom: 20px;
        }

        .empty-btn {
            display: inline-block;
            padding: 12px 24px;
            background: linear-gradient(135deg, #00aff5, #10b981);
            color: white;
            border-radius: 12px;
            font-size: 13px;
            font-weight: 700;
            text-decoration: none;
        }

        @media (max-width: 400px) {
            .search-row { grid-template-columns: 1fr; }
            .search-row.triple { grid-template-columns: 1fr 1fr; }
        }
    </style>
@endsection

@section('content')
    <!-- Search Header -->
    <div class="search-header">
        <h1 class="search-title">🔍 Find a Ride</h1>
        <p class="search-subtitle">Search for available rides across Pakistan</p>
    </div>

    <!-- Search Form -->
    <div class="search-card">
        <form action="{{ route('rides.search') }}" method="GET" class="search-form">
            <div class="search-row">
                <div class="form-group">
                    <label class="form-label">From</label>
                    <select name="from" class="form-select">
                        <option value="">Any city</option>
                        @foreach($cities ?? [] as $city)
                            <option value="{{ $city->id }}" {{ request('from') == $city->id ? 'selected' : '' }}>{{ $city->name }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">To</label>
                    <select name="to" class="form-select">
                        <option value="">Any city</option>
                        @foreach($cities ?? [] as $city)
                            <option value="{{ $city->id }}" {{ request('to') == $city->id ? 'selected' : '' }}>{{ $city->name }}</option>
                        @endforeach
                    </select>
                </div>
            </div>
            <div class="search-row triple">
                <div class="form-group">
                    <label class="form-label">Date</label>
                    <input type="date" name="date" class="form-input" value="{{ request('date', date('Y-m-d')) }}" min="{{ date('Y-m-d') }}">
                </div>
                <div class="form-group">
                    <label class="form-label">Seats</label>
                    <select name="seats" class="form-select">
                        <option value="">Any</option>
                        <option value="1" {{ request('seats') == '1' ? 'selected' : '' }}>1+</option>
                        <option value="2" {{ request('seats') == '2' ? 'selected' : '' }}>2+</option>
                        <option value="3" {{ request('seats') == '3' ? 'selected' : '' }}>3+</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">&nbsp;</label>
                    <button type="submit" class="search-btn">Search</button>
                </div>
            </div>
        </form>
    </div>

    <!-- Filters -->
    <div class="filter-row">
        <button class="filter-chip active">All</button>
        <button class="filter-chip">🕐 Morning</button>
        <button class="filter-chip">☀️ Afternoon</button>
        <button class="filter-chip">🌙 Evening</button>
        <button class="filter-chip">✓ Verified</button>
        <button class="filter-chip">❄️ AC</button>
    </div>

    @if($rides->count() > 0)
        <!-- Results Header -->
        <div class="results-header">
            <span class="results-count">{{ $rides->total() }} rides found</span>
            <button class="sort-btn">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                    <path d="M3 6h18"></path>
                    <path d="M6 12h12"></path>
                    <path d="M9 18h6"></path>
                </svg>
                Sort
            </button>
        </div>

        <!-- Rides List -->
        @foreach($rides as $ride)
            <a href="{{ route('bookings.create', ['ride' => $ride->id]) }}" class="ride-card">
                <div class="ride-image">
                    @if($ride->vehicle->vehicle_images && count($ride->vehicle->vehicle_images) > 0)
                        <img src="{{ asset('storage/' . $ride->vehicle->vehicle_images[0]) }}" alt="{{ $ride->vehicle->brand }}">
                    @else
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                            <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9C18.7 10.6 16 10 16 10s-1.3-1.4-2.2-2.3c-.5-.4-1.1-.7-1.8-.7H5c-.6 0-1.1.4-1.4.9l-1.4 2.9A3.7 3.7 0 0 0 2 12v4c0 .6.4 1 1 1h2"></path>
                            <circle cx="7" cy="17" r="2"></circle>
                            <circle cx="17" cy="17" r="2"></circle>
                        </svg>
                    @endif
                    <span class="ride-price-badge">Rs {{ number_format($ride->price_per_seat * 280, 0) }}</span>
                    <span class="ride-date-badge">{{ $ride->departure_datetime->format('D, M d') }}</span>
                </div>

                <div class="ride-content">
                    <div class="ride-route">
                        <div class="route-point">
                            <div class="route-city">{{ $ride->route->fromCity->name }}</div>
                            <div class="route-time">{{ $ride->departure_datetime->format('h:i A') }}</div>
                        </div>
                        <div class="route-arrow">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                                <path d="M5 12h14"></path>
                                <path d="M12 5l7 7-7 7"></path>
                            </svg>
                            <span class="route-duration">{{ $ride->route->estimated_duration_minutes ? floor($ride->route->estimated_duration_minutes / 60) . 'h' : '~2h' }}</span>
                        </div>
                        <div class="route-point" style="text-align: right;">
                            <div class="route-city">{{ $ride->route->toCity->name }}</div>
                            <div class="route-time">{{ $ride->arrival_datetime->format('h:i A') }}</div>
                        </div>
                    </div>

                    <div class="driver-row">
                        <div class="driver-avatar">
                            @if($ride->driver->avatar)
                                <img src="{{ asset('storage/' . $ride->driver->avatar) }}" alt="{{ $ride->driver->name }}">
                            @else
                                {{ substr($ride->driver->name, 0, 1) }}
                            @endif
                        </div>
                        <div class="driver-info">
                            <div class="driver-name">{{ $ride->driver->name }}</div>
                            <div class="driver-meta">
                                <span class="driver-rating">⭐ {{ number_format($ride->driver->rating ?? 4.5, 1) }}</span>
                                @if($ride->driver->is_verified)
                                    <span class="driver-verified">✓ Verified</span>
                                @endif
                                <span>{{ $ride->vehicle->brand }} {{ $ride->vehicle->model }}</span>
                            </div>
                        </div>
                        <span class="seats-badge">{{ $ride->available_seats }} seats</span>
                    </div>

                    @if($ride->amenities && count($ride->amenities) > 0)
                        <div class="amenities-row">
                            @foreach(array_slice($ride->amenities, 0, 4) as $amenity)
                                <span class="amenity-tag">{{ $amenity }}</span>
                            @endforeach
                        </div>
                    @endif
                </div>
            </a>
        @endforeach

        <!-- Pagination -->
        <div style="margin-top: 16px;">
            {{ $rides->appends(request()->query())->links() }}
        </div>
    @else
        <!-- Empty State -->
        <div class="empty-state">
            <div class="empty-icon">🚗</div>
            <h2 class="empty-title">No rides found</h2>
            <p class="empty-text">No rides match your search criteria.<br>Try different dates or cities.</p>
            <a href="{{ route('home') }}" class="empty-btn">Back to Home</a>
        </div>
    @endif
@endsection
